#!/bin/bash
# Copyright (C) 1999-2004,2007-2009 Red Hat, Inc. All rights reserved. This
# copyrighted material is made available to anyone wishing to use, modify,
# copy, or redistribute it subject to the terms and conditions of the
# GNU General Public License version 2.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# X Input method setup script

function log_init() {
    if [ ! -n "$DRY_RUN" ]; then
	rm -rf $HOME/.imsettings.log
	touch $HOME/.imsettings.log
    fi
}

function log() {
    echo "$@" >> $HOME/.imsettings.log
}

function is_dbus_enabled() {
    [ -n "$DBUS_SESSION_BUS_ADDRESS" ]
}

function is_imsettings_enabled() {
    [ -z "$DISABLE_IMSETTINGS" ] && is_dbus_enabled
}

function is_gtk_supported() {
    [ -n "$IMSETTINGS_DISABLE_DESKTOP_CHECK" ] && return 0
    case "$DESKTOP_SESSION" in
	*gnome|LXDE|openbox|xfce4)
	    return 0
	    ;;
	*)
	    return 1
	    ;;
    esac
}

function is_qt_supported() {
    [ -n "$IMSETTINGS_DISABLE_DESKTOP_CHECK" ] && return 0
    case "$DESKTOP_SESSION" in
	*)
	    return 1
	    ;;
    esac
}

function setup_gtk_immodule() {
    if is_imsettings_enabled && is_gtk_supported; then
	# Ensure GTK_IM_MODULE is empty. otherwise GTK+ doesn't pick up immodule through XSETTINGS
	unset GTK_IM_MODULE
	export GTK_IM_MODULE
    else
	[ -n "$GTK_IM_MODULE" ] && export GTK_IM_MODULE
    fi
}

function setup_qt_immodule() {
    if is_imsettings_enabled && is_qt_supported; then
	# FIXME: Qt doesn't support XSETTINGS for immodule yet.
	#        We still need to go with the older way.
	[ -n "$QT_IM_MODULE" ] && export QT_IM_MODULE
    else
	[ -n "$QT_IM_MODULE" ] && export QT_IM_MODULE
    fi
}

function setup_xim() {
# XXX: Disable XIM support so far.
#    if is_imsettings_enabled; then
#	# setup XMODIFIERS
#	XMODIFIERS="@im=imsettings"
#	export XMODIFIERS
#    else
	[ -z "$XMODIFIERS" -a -n "$XIM" ] && XMODIFIERS="@im=$XIM"
	[ -n "$XMODIFIERS" ] && export XMODIFIERS
#    fi
}

function run_imsettings() {
    print_info
    if [ -n "$DRY_RUN" ]; then
	log "*** DRY RUN MODE: running IM through imsettings"
    else
	if [ -n "$IMSETTINGS_INTEGRATE_DESKTOP" -a "x$IMSETTINGS_INTEGRATE_DESKTOP" = "xno" ]; then
	    which imsettings-start > /dev/null 2>&1 && LANG="$tmplang" imsettings-start -n "$IMSETTINGS_MODULE" || :
	    # NOTE: We don't bring up imsettings-xim nor imsettings-applet here to support XIM.
	    #       imsettings-applet will starts through XDG autostart mechanism.
	    #       If the desktop doesn't support that, this function shouldn't be invoked.
	    #       but run_xim() instead.
	fi
    fi
}

function run_xim() {
    print_info
    if [ -n "$DRY_RUN" ]; then
	log "*** DRY RUN MODE: running IM without imsettings"
    else
	DISABLE_IMSETTINGS=true
	export DISABLE_IMSETTINGS

	# execute XIM_PROGRAM
	[ -n "$XIM_PROGRAM" ] && which "$XIM_PROGRAM" > /dev/null 2>&1 && LANG="$tmplang" "$XIM_PROGRAM" $XIM_ARGS > $HOME/.imsettings.log 2>&1 &
    fi
}

function print_result() {
    $1
    if [ $? -eq 0 ]; then
	log yes
    else
	log no
    fi
}

function print_info() {
    log "imsettings information"
    log "=========================="
    log -n "Is DBus enabled: "
    print_result is_dbus_enabled
    log -n "Is imsettings enabled: "
    print_result is_imsettings_enabled
    log -n "Is GTK+ supported: "
    print_result is_gtk_supported
    log -n "Is Qt supported: "
    print_result is_qt_supported
    log "DESKTOP_SESSION: $DESKTOP_SESSION"
    log "DISABLE_IMSETTINGS: $DISABLE_IMSETTINGS"
    log "IMSETTINGS_DISABLE_DESKTOP_CHECK: $IMSETTINGS_DISABLE_DESKTOP_CHECK"
    log "DBUS_SESSION_BUS_ADDRESS: $DBUS_SESSION_BUS_ADDRESS"
    log "GTK_IM_MODULE: $GTK_IM_MODULE"
    log "QT_IM_MODULE: $QT_IM_MODULE"
    log "XMODIFIERS: $XMODIFIERS"
    log "IMSETTINGS_MODULE: $IMSETTINGS_MODULE"
    log "IMSETTINGS_INTEGRATE_DESKTOP: $IMSETTINGS_INTEGRATE_DESKTOP"
    log ""
}

USER_XINPUTRC="$HOME/.xinputrc"
SYS_XINPUTRC="/etc/X11/xinit//xinputrc"

# Load up the user and system locale settings
oldterm=$TERM
unset TERM
if [ -r /etc/profile.d/lang.sh ]; then
    # for Fedora etc
    source /etc/profile.d/lang.sh
elif [ -r /etc/default/locale ]; then
    # for Debian
    source /etc/default/locale
fi
[ -n "$oldterm" ] && export TERM=$oldterm

tmplang=${LC_CTYPE:-${LANG:-"en_US.UTF-8"}}

# unset env vars to be safe
unset AUXILIARY_PROGRAM AUXILIARY_ARGS GTK_IM_MODULE ICON IMSETTINGS_IGNORE_ME LONG_DESC PREFERENCE_PROGRAM PREFERENCE_ARGS QT_IM_MODULE SHORT_DESC XIM XIM_PROGRAM XIM_ARGS XMODIFIERS

[ -z "$IMSETTINGS_DISABLE_USER_XINPUTRC" ] && IMSETTINGS_DISABLE_USER_XINPUTRC=no

if [ -r "$USER_XINPUTRC" -a "x$IMSETTINGS_DISABLE_USER_XINPUTRC" = "xno" ]; then
    source "$USER_XINPUTRC"
    if [ ! -h "$USER_XINPUTRC" ]; then
	SHORT_DESC="User Specific"
    fi
elif [ -r "$SYS_XINPUTRC" ]; then
    # FIXME: This hardcoded list has to be gone in the future.
    # Locales that normally use input-method for native input
    _im_language_list="as bn gu hi ja kn ko mai ml mr ne or pa si ta te th ur vi zh"
    _sourced_xinputrc=0
    for i in $_im_language_list; do
        if echo $tmplang | grep -q -E "^$i"; then
            source "$SYS_XINPUTRC"
            _sourced_xinputrc=1
            break
        fi
    done
    # Locales that usually use X locale compose
    # FIXME: which other locales should be included here?
    if [ $_sourced_xinputrc -eq 0 ]; then
        _xcompose_language_list="am_ET el_GR fi_FI pt_BR ru_RU"
        for i in $_xcompose_language_list; do
            if echo $tmplang | grep -q -E "^$i"; then
                source /etc/X11/xinit/xinput.d//xcompose.conf
                _sourced_xinputrc=1
                break
            fi
        done
    fi
    if [ $_sourced_xinputrc -eq 0 ]; then
        # Read none.conf to set up properly for locales not listed the above.
        source /etc/X11/xinit/xinput.d//none.conf
    fi
fi

[ -z "$IMSETTINGS_INTEGRATE_DESKTOP" ] && IMSETTINGS_INTEGRATE_DESKTOP=yes
export IMSETTINGS_INTEGRATE_DESKTOP

[ -z "$XIM" ] && XIM=none

# start IM via imsettings
IMSETTINGS_MODULE=${SHORT_DESC:-${XIM}}
[ -z "$IMSETTINGS_MODULE" ] && IMSETTINGS_MODULE="none"
export IMSETTINGS_MODULE

##
log_init
setup_gtk_immodule
setup_qt_immodule
setup_xim

# NOTE: Please make sure the session bus is established before running this script.
if ! is_dbus_enabled; then
    log "***"
    log "*** No DBus session hasn't been established yet. giving up to deal with Input Method with imsettings."
    log "***"

    run_xim
elif ! is_imsettings_enabled; then
    log "***"
    log "*** imsettings is explicitly disabled."
    log "***"

    run_xim
else
    # Yes, we are in the dbus session
    run_imsettings
fi
